package dressing.ui.preferences;

import java.util.*;
import java.util.List;
import java.util.stream.*;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

import dressing.config.ReportingPreferences;

public class DynamicPreferencesDialog extends TitleAreaDialog {

	private final Map<String,String> labels;
    private List<String> groups;
    private Map<String,List<String>> groupToKeys;
    private TreeViewer treeViewer;
    private Composite pageContainer;
    private List<Control > currentPageCheckboxes = new ArrayList<Control>();
    private Map<String, Boolean> pageInitialState = new HashMap<String, Boolean>();
    private String currentGroup;

    public DynamicPreferencesDialog(Shell parentShell, Map<String,String> labels) {
        super(parentShell);
        this.labels = labels;
        setTitle(currentGroup);
        buildModel();
    }
    @Override
    public void create() {
        super.create();
        setTitle("Préférences");
        setMessage("Personnalisez vos Préférences");
        getShell().setMaximized(true);
        getShell().setText("Préférences");
    }
    // Allow maximize/resize
    @Override
    protected int getShellStyle() {
        return super.getShellStyle() | SWT.RESIZE | SWT.MAX;
    }

    private void buildModel() {
        Properties props = ReportingPreferences.getInstance();
        groupToKeys = new TreeMap<>();
        for (String key : props.stringPropertyNames()) {
            String group = key.contains(".") ? key.substring(0, key.indexOf('.')) : key;
            // Skip 'user' group
            if ("user".equals(group)) continue;
            groupToKeys.computeIfAbsent(group, g -> new ArrayList<>()).add(key);
        }
        // Remove any existing 'user' group if present
        groupToKeys.remove("user");
        groups = new ArrayList<>(groupToKeys.keySet());
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        setTitle("Préférences");
        setMessage("Personnalisez vos options de rapport.");

        SashForm sash = new SashForm(parent, SWT.HORIZONTAL);
        sash.setLayoutData(new GridData(GridData.FILL_BOTH));

        // Left: simple tree with groups only
        treeViewer = new TreeViewer(sash, SWT.BORDER | SWT.SINGLE);
        treeViewer.getTree().setLayoutData(new GridData(GridData.FILL_BOTH));
        treeViewer.setContentProvider(new ITreeContentProvider() {
            @Override public Object[] getElements(Object input) {
                return groups.toArray();
            }
            @Override public Object[] getChildren(Object element) { return new Object[0]; }
            @Override public Object getParent(Object element) { return null; }
            @Override public boolean hasChildren(Object element) { return false; }
        });
        treeViewer.setLabelProvider(new LabelProvider() {
            @Override public String getText(Object element) {
                String g = (String)element;
                return labels.getOrDefault(g, g);
            }
        });
        treeViewer.setInput(groups);

        // Right: container for checkboxes
        pageContainer = new Composite(sash, SWT.NONE);
        pageContainer.setLayout(new GridLayout(1, false));
        pageContainer.setLayoutData(new GridData(GridData.FILL_BOTH));

        sash.setWeights(new int[]{1, 3});

        // on select group
        treeViewer.addSelectionChangedListener(evt -> {
            IStructuredSelection sel = treeViewer.getStructuredSelection();
            if (!sel.isEmpty()) {
                String newGroup = (String)sel.getFirstElement();
                if (currentGroup != null && isPageDirty()) {
                    int choice = new MessageDialog(getShell(),
                        "Enregistrer les modifications ?", null,
                        "Modifications non enregistrées pour '" + labels.getOrDefault(currentGroup, currentGroup) + "'. Enregistrer ?",
                        MessageDialog.QUESTION,
                        new String[]{"Enregistrer", "Ignorer", "Annuler"}, 0).open();
                    if (choice == 2) {
                        treeViewer.setSelection(new StructuredSelection(currentGroup));
                        return;
                    } else if (choice == 0) {
                        saveCurrentPage();
                    }
                }
                showGroupPage(newGroup);
                currentGroup = newGroup;
            }
        });

        // init
        if (!groups.isEmpty()) {
            String first = groups.get(0);
            treeViewer.setSelection(new StructuredSelection(first));
            currentGroup = first;
            showGroupPage(first);
        }

        return sash;
    }

    private boolean isPageDirty() {
        for (Control control : currentPageCheckboxes) {
            String key = (String)control.getData("prefKey");
            String original = ReportingPreferences.getInstance().getProperty(key, "");
            String current;
            if (control instanceof Button) {
                current = Boolean.toString(((Button) control).getSelection());
            } else if (control instanceof Text) {
                current = ((Text) control).getText();
            } else {
                continue;
            }
            if (!Objects.equals(original, current)) {
                return true;
            }
        }
        return false;
    }

    private void showGroupPage(String group) {
        for (Control c : pageContainer.getChildren()) c.dispose();
        currentPageCheckboxes.clear();
        pageInitialState.clear();

        // Header
        Composite header = new Composite(pageContainer, SWT.NONE);
        header.setLayout(new GridLayout(3, false));
        header.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        Label title = new Label(header, SWT.NONE);
        title.setText(labels.getOrDefault(group, group));
        title.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false));

        Button selectAll = new Button(header, SWT.PUSH);
        selectAll.setText("Tout cocher");
        selectAll.addListener(SWT.Selection, e -> {
            for (Control cb : currentPageCheckboxes) {
                if (cb instanceof Button && (cb.getStyle() & SWT.CHECK) != 0) {
                    ((Button) cb).setSelection(true);
                }
            }
        });

        Button deselectAll = new Button(header, SWT.PUSH);
        deselectAll.setText("Tout décocher");
        deselectAll.addListener(SWT.Selection, e -> {
            for (Control cb : currentPageCheckboxes) {
                if (cb instanceof Button && (cb.getStyle() & SWT.CHECK) != 0) {
                    ((Button) cb).setSelection(false);
                }
            }
        });

        // checkboxes
        ReportingPreferences prefs = ReportingPreferences.getInstance();
        for (String key : groupToKeys.getOrDefault(group, Collections.emptyList())) {
            if (ReportingPreferences.isBooleanProperty(key)) {
                boolean sel = Boolean.parseBoolean(prefs.getProperty(key, "false"));
                Button cb = new Button(pageContainer, SWT.CHECK);
                cb.setText(labels.getOrDefault(key, key));
                cb.setSelection(sel);
                cb.setData("prefKey", key);
                currentPageCheckboxes.add(cb);
                pageInitialState.put(key, sel);
            } else {
                Composite row = new Composite(pageContainer, SWT.NONE);
                row.setLayout(new GridLayout(2, false));
                row.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

                Label lbl = new Label(row, SWT.NONE);
                lbl.setText(labels.getOrDefault(key, key));
                lbl.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));

                Text txt = new Text(row, SWT.BORDER);
                txt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
                txt.setText(prefs.getProperty(key, ""));
                txt.setData("prefKey", key);
                currentPageCheckboxes.add(txt); // still add to list
            }
        }

        pageContainer.layout(true, true);
    }

    @Override
    protected void okPressed() {
        if (isPageDirty()) saveCurrentPage();
        super.okPressed();
    }

    private void saveCurrentPage() {
        ReportingPreferences prefs = ReportingPreferences.getInstance();
        for (Control control : currentPageCheckboxes) {
            String key = (String)control.getData("prefKey");
            String value;
            if (control instanceof Button) {
                value = Boolean.toString(((Button) control).getSelection());
            } else if (control instanceof Text) {
                value = ((Text) control).getText();
            } else {
                continue;
            }
            prefs.setProperty(key, value);
            pageInitialState.put(key, Boolean.valueOf(value));
        }
        prefs.save();
    }

}
