package dressing.mathutils;

import java.util.ArrayList;
import java.util.Random;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import gdxapp.assets.AssetFont;
import gdxapp.screens.room.RoomController;
import geometry.Shape;

public class Edge implements Shape{
	
    private static ShapeRenderer shapeRenderer;
    private static BitmapFont font;

	private Vector2 v0;
	private Vector2 v1;
	private boolean isSelected;
	private float[] constraint;
	
	public Edge() {}
	
	public Edge(Vector2 start, Vector2 end) {
		super();
		this.v0 = start;
		this.v1 = end;
	}
		

	//getters and setters
	public Vector2 getV0() {
		return v0;
	}

	public void setV0(Vector2 start) {
		this.v0 = start;
	}

	public Vector2 getV1() {
		return v1;
	}

	public void setV1(Vector2 end) {
		this.v1 = end;
	}

	public Vector2 getMiddle() {
		return v0.cpy().add(v1).scl(0.5f);
	}

	
	
	public boolean isSelected() {
		return isSelected;
	}

	public void setSelected(boolean isSelected) {
		this.isSelected = isSelected;
	}
	
    public Vector2 getDirector() {
		return v1.cpy().sub(v0).nor();
	}

	public Vector2 getNormal(){
		return v1.cpy().sub(v0).rotate(90).nor();
	}

	
	
	 public float[] getConstraint() {
		return constraint;
	}

	public Edge getMedian(){
	        Vector2 origin = v1.cpy().add(v0).scl(0.5f);
	        return new Edge(origin, origin.cpy().add(getNormal()));
	    }

	public void draw(Batch batch, float parentAlpha) {
		if(v0 != null && v1 != null) {
			if(shapeRenderer == null)
				shapeRenderer = new ShapeRenderer();
			if(font == null)
				font = AssetFont.getInstance().getFont();
	        shapeRenderer.setProjectionMatrix(batch.getProjectionMatrix());
	        shapeRenderer.setColor(Color.BLACK);
	        if(isSelected)
	        	shapeRenderer.setColor(Color.RED);
	        if(batch.isDrawing())
	        	batch.end();
	        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
	        shapeRenderer.rectLine(v0.x, v0.y, v1.x, v1.y, 3);
	        shapeRenderer.setColor(Color.GREEN);
	        var middle = getMiddle();
	        var normal = getNormal();
	        shapeRenderer.rectLine(middle.x, middle.y,middle.x + normal.x, middle.y + normal.y, 3);
	        shapeRenderer.end();
	        batch.begin();
	        font.setColor(Color.RED);
	        font.draw(batch,"" + (int)((getLength() * 1000) / RoomController.getInstance().getScaleX()),getMiddle().x,getMiddle().y );
	        batch.end();
		}
		
    }
	
	public void draw(Batch batch, float parentAlpha, Color color) {
		if(v0 != null && v1 != null) {
			if(shapeRenderer == null)
				shapeRenderer = new ShapeRenderer();
	        shapeRenderer.setProjectionMatrix(batch.getProjectionMatrix());
	        shapeRenderer.setColor(color);
	        if(batch.isDrawing())
	        	batch.end();
	        draw(shapeRenderer, color);
	        batch.begin();
		}		
	}
    public float getLength() {
        return v1.cpy().sub(v0).len();
    }
    
	public void setLength(float length, int direction) {
		switch(direction) {
		case 0:
			// direction = 0 , resize and keep the center
			this.v0.set(getMiddle().sub(getDirector().scl(0.5f * length)));
			this.v1.set(getMiddle().add(getDirector().scl(0.5f * length)));
			break;
		case 1:
			//direction = 1 , resize and keep the start point
			this.v1.set(this.v0.cpy().add(getDirector().scl(length)));
			break;
		case -1:
			//directop = -1, resize and keep the end point
			this.v0.set(this.v1.cpy().sub(getDirector().scl(length)));
		}
	}
	
	public boolean hasVertex(Vector2 vertex) {
		if(this.v0.epsilonEquals(vertex,0.1f) || this.v1.epsilonEquals(vertex,0.1f))
			return true;
		return false;
	}
	public boolean contains(Vector2 point) {
		return Math.abs(point.dst(v0) + point.dst(v1) - v0.dst(v1)) < 0.01f;
	}
	
	public float getRotation() {
		return v1.cpy().sub(v0).angle();
	}
	
	boolean intersects(Edge other) {
		//to do
		return false;
	}
	
	@Override
	public void draw(ShapeRenderer shapeRenderer, Color color) {
		 shapeRenderer.setColor(color);
		 shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
	     shapeRenderer.rectLine(v0.x, v0.y, v1.x, v1.y, 5);
	     shapeRenderer.end();
	     shapeRenderer.setColor(Color.BLACK);
	}

	@Override
	public ArrayList<Vector2> getVertices() {
		ArrayList<Vector2> vertices = new ArrayList<Vector2>();
		vertices.add(v0);
		vertices.add(v1);
		return vertices;
	}

	@Override
	public Vector2 getlastVertex() {
		return v1;
	}

	@Override
	public void scale(Vector2 scale) {
		this.v0.scl(scale);
		this.v1.scl(scale);
	}

	@Override
	public Shape cpy() {
		return new Edge(v0.cpy(),v1.cpy());
	}

	@Override
	public float distanceTo(Vector2 point) {
		Vector2 projectedPoint = MathUtilities.projectPoint(point, v0, v1);
		Vector2 v0m = projectedPoint.cpy().sub(this.v0);
		Vector2 v1m = projectedPoint.cpy().sub(v1);
		if(!v0m.hasOppositeDirection(v1m))
		{
			return Math.min(point.dst(v0), point.dst(v1));
		}else {
			return point.dst(projectedPoint);
		}	
	}

	@Override
	public Vector2 projectPoint(Vector2 point) {
		if(contains(point))
			return new Vector2(point);
		Vector2 projectedPoint = MathUtilities.projectPoint(point, v0, v1);
		Vector2 v0m = projectedPoint.cpy().sub(this.v0);
		Vector2 v1m = projectedPoint.cpy().sub(v1);
		if(v0m.hasOppositeDirection(v1m))
		{
			return projectedPoint;
		}else {
			return projectedPoint.dst(v0) < projectedPoint.dst(v1)?v0.cpy():v1.cpy();
		}	
	}

	public void addConstraint(Vector2 center, float width, float height) {
		constraint = new float[] {center.x, center.y, width, height};
		
	}
	
	
	@Override
	public String toString() {
		return "Edge [v0=" + v0 + ", v1=" + v1 + "]";
	}
	
	

	@Override
	public int hashCode() {
		float length = getLength();
		int exp = (int) Math.log10(length);
		float epsilon = (float) Math.pow(10, exp - 2);
		int qx = Math.round(v0.x / epsilon);
		int qy = Math.round(v0.y / epsilon);
		int qz = Math.round(v1.x / epsilon);
		int qw = Math.round(v1.y / epsilon);
		int hash = 17;
		hash = 31 * hash + qx;
		hash = 31 * hash + qy;
		hash = 31 * hash + qz;
		hash = 31 * hash + qw;
		return hash;
	}

	@Override
	public boolean equals(Object arg0) {
		if(this == arg0)
			return true;
		if(!(arg0 instanceof Edge))
			return false;
		Edge other = (Edge) arg0;
		float length = getLength();
		float otherLength = other.getLength();
		if(Math.abs(length/otherLength - 1) > 1e-2f)
			return false;
		int exp = (int) Math.log10(length);
		float epsilon = (float) Math.pow(10, exp - 2);
		if(getV0().epsilonEquals(other.getV0(), epsilon) && getV1().epsilonEquals(other.getV1(), epsilon))
			return true;
		return false;
	}

	@Override
	public Object select(Vector2 position) {
		Vector2 projectedPoint = projectPoint(position);
		if(projectedPoint.dst(position) > 20)
			return null;
		float distanceToV0 = v0.dst(projectedPoint), distanceToV1 = v1.dst(projectedPoint), distanceToMiddle = getMiddle().dst(projectedPoint);
		float minDistance = Math.min(Math.min(distanceToV0, distanceToV1),distanceToMiddle);
		Object selection = null;
		if(minDistance < 100) {
			if(minDistance == distanceToV0) {
				selection = v0;
			}else if(minDistance == distanceToV1) {
				selection = v1;
			}else {
				selection = this;
			}
		}
		return selection; 
	}

	public void translate(float xmov, float ymov) {
		this.v0.add(xmov,ymov);
		this.v1.add(xmov, ymov);
	}

	@Override
	public int getWinding() {
		return 0;
	}


	public void flip(){
		this.v0.add(this.v1);
		this.v1.set(v0.cpy().sub(v1));
		this.v0.sub(v1);
	}

	@Override
	public void setWinding(int winding) {
		//do nothing
	}
	
	
	public static void main(String...strings) {
		Random random = new Random();
		
		float epsilon = 1e-1f;
		
		float a = random.nextFloat() * 1400;
		float b = random.nextFloat() * 1400;
		
		float c = random.nextFloat() * 1400;
		float d = random.nextFloat() * 1400;
		
		Vector2 v0 = new Vector2(a, b);
		Vector2 v1= new Vector2(c, d);

		Edge edge = new Edge(v0, v1);
		Edge edge2 = new Edge(v0.cpy().add(epsilon,epsilon), v1);
		System.out.println("edge1: " +  edge.hashCode());
		System.out.println("edge2: " +  edge2.hashCode());

				
	}

	public boolean epsilonEquals(Edge other, float epsilon, boolean sameDir) {
		if(v0.epsilonEquals(other.getV0(), epsilon) && v1.epsilonEquals(other.getV1(), epsilon))
			return true;
		if( !sameDir && (v0.epsilonEquals(other.getV1(), epsilon) && v1.epsilonEquals(other.getV0(), epsilon) ) )
			return true;
		return false;

	}
	
	
}
