package dressing.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.widgets.Display;

import dressing.config.WorkspaceConfiguration;
import dressing.model.ModelProvider;
import dressing.ui.ProgressBarMonitor;
import dressing.ui.engine3d.SceneTexture;
import dressing.ui.engine3d.SceneTexture.CATEGORY;

public class ImageLabelLoader implements Runnable {
		
	public ImageLabelLoader() {
		super();
	}

	@Override
	public void run() {
		List<SceneTexture> textures = ModelProvider.getTextures();
		List<SceneTexture> texturesWithoutView = textures.stream()
			.filter(t -> t.getCategory() != CATEGORY.VIEW)
			.toList();

		// Combined total for progress
		int totalSteps = textures.size() + texturesWithoutView.size();
		Display.getDefault().syncExec(() -> ProgressBarMonitor.setTasks(totalSteps));

		// Step 1: Generate preview and label images if missing
		for (SceneTexture texture : textures) {
			File imageFile = new File(texture.getPath());

			if (!imageFile.exists() || !imageFile.isFile()) {
				System.err.println("Skipping non-file: " + imageFile.getAbsolutePath());
				continue;
			}

			
			File minifiedFile = new File(texture.getMinifiedPath());
			File labelFile = new File(texture.getLabelPath());

			if (!minifiedFile.exists() || !labelFile.exists()) {
				try {
					ImageData fullData = new ImageData(texture.getPath());
					ImageData previewData = fullData.scaledTo(256, 256);
					ImageData labelData = fullData.scaledTo(128, 32);
					IOUtilities.saveImageDataToPNG(previewData, minifiedFile);
					IOUtilities.saveImageDataToPNG(labelData, labelFile);
				} catch (Exception e) {
					System.err.println("Failed to load image: " + texture.getPath());
					e.printStackTrace();
					continue;
				}
			}

			String msg = "Mise en cache " + new File(texture.getPath()).getName();
			Display.getDefault().syncExec(() -> {
				ProgressBarMonitor.advance(1);
				ProgressBarMonitor.labelTask(msg);
			});
		}
		// Step 2: Load SWT Images for UI use
		Display.getDefault().syncExec(() -> {
			for (SceneTexture texture : texturesWithoutView) {
				try {
					File f = new File(texture.getPath());
					if(f.exists() && f.isFile()) {
						Image preview = new Image(Display.getDefault(), texture.getMinifiedPath());
						Image label = new Image(Display.getDefault(), texture.getLabelPath());
						texture.setPreview(preview);
						texture.setLabelImage(label);

						String msg = "Chargée " + new File(texture.getPath()).getName();
						ProgressBarMonitor.advance(1);
						ProgressBarMonitor.labelTask(msg);
					}else {
						
						String msg = "Ignorer " + new File(texture.getPath()).getName();
						ProgressBarMonitor.advance(1);
						ProgressBarMonitor.labelTask(msg);
						System.err.println("Ignore loading  "+texture.getPath());
					}

				}catch (Exception e) {
					e.printStackTrace();
				}

			}
		});
	}
	


}
