package gdxapp.object3d;

import java.util.ArrayList;
import java.util.List;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.VertexAttribute;
import com.badlogic.gdx.graphics.VertexAttributes.Usage;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelBatch;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.Vector2;

import com.badlogic.gdx.math.collision.BoundingBox;
import dressing.mathutils.MathUtilities;
import dressing.model.types.PoigneeType;
import gdxapp.assets.AssetFont;
import gdxapp.object3d.WorldObject.ObjectType;
import gdxapp.quotation.Object3DQuotation;
import org.frs.svg.LineSegment;

public class Object3D extends ModelInstance {

	private WorldObject worldObject;
	private final Vector3 error = new Vector3(0, 0, 0);
	private final Vector3 rectification = new Vector3();
	private final BoundingBox boundingBox = new BoundingBox();
	private final RotatedBoundingBox rotatedBoundingBox = new RotatedBoundingBox();
	float radius;
	private  Vector3 position = new Vector3();
	private  Vector3 dimensions = new Vector3();
	float depthRectification = 0.0f;
	private Model sketchModel;
	private Object3D sketchInstance;
	private float optimalVolumeAngle;
	private Mesh boundingMesh;
	private static ShapeRenderer shapeRenderer;
	private List<Object3D> addons = new ArrayList<Object3D>();
	

	public Object3D(Model model) {
		super(model);
		calculateProperties();
	}

	public Object3D(WorldObject worldObject) {
		super(worldObject.getModel().getModel());
		this.worldObject = worldObject;
		Vector3 dimension = new Vector3();
		calculateBoundingBox(boundingBox);
		boundingBox.getDimensions(dimension);
		Vector3 dimensionTargetted = getExteriorDimension(worldObject);
		if (worldObject.getType() != ObjectType.DEFINED) {
			for (int i = 0; i < nodes.size; i++) {
				nodes.get(i).scale.set(dimensionTargetted.x / dimension.x, dimensionTargetted.y / dimension.y,
						dimensionTargetted.z / dimension.z);
			}
			calculateTransforms();
			calculateBoundingBox(boundingBox);
		}
		try {
			boundingBox.getCenter(error);
		} catch (Exception e) {
			error.set(0, 0, 0);
		}
		transform.setToTranslation(-error.x, -error.y, -error.z);

		if (worldObject.getRealWorldPosition() != null) {
			depthRectification = dimensionTargetted.z - worldObject.getRealWorldDimension().z;
			depthRectification /= 2.f;
			rectification.set(0, 0, depthRectification);
			rectification.rotate(Vector3.Y, worldObject.getRotation());
			transform.translate(worldObject.getRealWorldPosition().cpy().add(rectification));
			rotate(Vector3.Y, worldObject.getRotation());
		}
		calculateBoundingBox(boundingBox);
		if (worldObject instanceof KitchenElement)
			createSketch();
	}

	private void createSketch() {
		this.sketchModel = DesignModelFactory.getFactory().createSketchFromVertices((KitchenElement) worldObject);
		this.sketchInstance = new Object3D(sketchModel);
		this.sketchInstance.transform.set(transform);
		this.sketchInstance.calculateProperties();
	}

	public void updatePosition() {
		Vector3 oldPosition = new Vector3();
		boundingBox.getCenter(oldPosition);
		Vector3 newPosition = worldObject.getRealWorldPosition().cpy().add(rectification) ;
		if(newPosition.epsilonEquals(oldPosition, 0.001f))
			return;
		Matrix4 oldTransform = transform == null ? new Matrix4()
				: transform.cpy();
		transform.setToTranslation(-error.x, -error.y, -error.z);
		if (worldObject.getRealWorldPosition() != null) {
			if (rotatedBoundingBox != null) {
				Vector3 rec = new Vector3(0, 0, depthRectification);
				rec.rotate(Vector3.Y, worldObject.getRotation());
				transform.translate(worldObject.getRealWorldPosition().cpy().add(rec));
				rotate(Vector3.Y, worldObject.getRotation());
				calculateProperties();
				Matrix4 transition = oldTransform.inv().cpy().mul(transform);
				Vector3 translation = new Vector3();
				Quaternion quat = new Quaternion();
				transition.getRotation(quat, true);
				Vector3 axis = new Vector3();
				quat.getAxisAngle(axis);
				transition.getTranslation(translation);
				float angle = quat.getAngle();
				Vector3 center = new Vector3();
				boundingBox.getCenter(center);

				calculateEffectiveBoundingBox();
			}
		}
		if (sketchInstance != null) {
			sketchInstance.transform.set(transform);
			sketchInstance.calculateProperties();
		}
	}

	public WorldObject getWorldObject() {
		return worldObject;
	}

	public void setWorldObject(WorldObject worldObject) {
		this.worldObject = worldObject;
	}

	public Vector3 getExteriorDimension(WorldObject worldObject) {
		Vector3 dims = new Vector3(worldObject.getRealWorldDimension().x, worldObject.getRealWorldDimension().y,
				worldObject.getRealWorldDimension().z);
		if (worldObject instanceof KitchenElement) {
			KitchenElement kitchenElement = (KitchenElement) worldObject;
			if (!kitchenElement.getDesignObject().getDesignCaissonType().contains("COINS")) {
				float facadeThickness = kitchenElement.getDesignObject().getFacadeThickness() * 0.001f;
				dims.z += facadeThickness;
			}
		}
		return dims;
	}

	public void calculateProperties() {
		calculateBoundingBox(boundingBox);
		boundingBox.getDimensions(this.dimensions);
		boundingBox.getCenter(this.position);
		radius = this.dimensions.cpy().scl(0.5f).len();
	}

	public boolean contains(Vector3 point) {
		if (position.dst(point) > radius)
			return false;
		Vector3 halfDims = dimensions.cpy().scl(0.5f);
		Vector3 lowerBound = new Vector3(position).sub(halfDims);
		Vector3 upperBound = new Vector3(position).add(halfDims);
		if (point.x < lowerBound.x || point.x > upperBound.x)
			return false;
		if (point.y < lowerBound.y || point.y > upperBound.y)
			return false;
		if (point.z < lowerBound.z || point.z > upperBound.z)
			return false;
		return true;
	}

	public boolean intersects(Object3D other) {
		return getRotatedBoundingBox().intersects(other.getRotatedBoundingBox());
	}
	
	public List<Object3DQuotation> getQuotations(){
		ArrayList<Object3DQuotation> quots = new ArrayList<Object3DQuotation>();
		Vector3 v000 = new Vector3();
		getRotatedBoundingBox().getCorner000(v000);
		//v000.mul(getRotatedBoundingBox().getTransform());
		Vector3 v001 = new Vector3();
		getRotatedBoundingBox().getCorner001(v001);
		//v001.mul(getRotatedBoundingBox().getTransform());
		Vector3 v101 = new Vector3();
		getRotatedBoundingBox().getCorner101(v101);
		//v101.mul(getRotatedBoundingBox().getTransform());
		Vector3 v011 = new Vector3();
		getRotatedBoundingBox().getCorner011(v011);
		//v011.mul(getRotatedBoundingBox().getTransform());
		boolean flippedXZ = Math.abs( v001.dst(v101) - worldObject.getRealWorldDimension().x ) > 0.01f;
		float xValue = flippedXZ?worldObject.getRealWorldDimension().z: worldObject.getRealWorldDimension().x;
		float zValue = worldObject.getRealWorldDimension().x + worldObject.getRealWorldDimension().z - xValue;
		
		quots.add(new Object3DQuotation(v001.cpy(), v101.cpy(), this, xValue));
		quots.add(new Object3DQuotation(v001.cpy(), v011.cpy(), this,worldObject.getRealWorldDimension().y));
		quots.add(new Object3DQuotation(v000.cpy(), v001.cpy(), this, zValue));
		return quots;
	}

	public void translate(Vector3 translation) {
		transform.translate(translation);
		calculateProperties();
	}

	public void rotate(Vector3 axis, float degree) {
		transform.translate(error.cpy()).rotate(axis, degree).translate(error.cpy().scl(-1));
	}

	public Vector3[] getCorners() {
		Vector3 corner0 = new Vector3();
		Vector3 corner1 = new Vector3();
		Vector3 corner2 = new Vector3();
		Vector3 corner3 = new Vector3();
		rotatedBoundingBox.getCorner000(corner0);
		rotatedBoundingBox.getCorner100(corner1);
		rotatedBoundingBox.getCorner010(corner2);
		rotatedBoundingBox.getCorner001(corner3);
		return new Vector3[] { corner0, corner1, corner2, corner3 };
	}

	public Vector3[] getBounds() {
		ArrayList<Vector3> corners = new ArrayList<Vector3>();

		corners.add(rotatedBoundingBox.getCorner000(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner001(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner010(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner011(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner100(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner101(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner110(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner111(new Vector3()));

		return MathUtilities.getBoundaries(corners);
	}


	public float getRadius() {
		calculateProperties();
		return radius;
	}

	public Vector3 getPosition() {
		calculateProperties();
		return position;
	}

	public Vector3 getDimension() {
		return dimensions.cpy();
	}

	boolean visible(Camera camera) {
		return false;
	}

	public void drawBoundaries(ShaderProgram shader, Matrix4 vp) {
		if (boundingMesh != null && rotatedBoundingBox!=null) {
			shader.begin();
			Gdx.gl.glEnable(GL20.GL_DEPTH_TEST);
			Gdx.gl.glDepthFunc(GL20.GL_LEQUAL);
			Gdx.gl.glLineWidth(2);
			float[] color = {1,0,0};
			if (vp != null)
				shader.setUniformMatrix("vp", vp);
			shader.setUniformf("hasAlbedoMap", 0.0f);
			shader.setUniform3fv("material.albedo", color, 0, 3);
			shader.setUniformMatrix("model", rotatedBoundingBox.getTransform());
			try{
				boundingMesh.render(shader, GL20.GL_LINES);
			}catch (Exception e) {
				System.out.println(e);
			}
			shader.end();
			Gdx.gl.glLineWidth(1);

		}
	}

	public Object3D getSketch() {
		return sketchInstance;
	}

	public void render(ModelBatch batch) {
		batch.render(this);
		if (sketchInstance != null)
			batch.render(sketchInstance);
		for(var addon: addons) {
			Matrix4 relativeTransform =  addon.transform.cpy();
			addon.transform.mulLeft(this.transform);
			addon.render(batch);
			addon.transform.set(relativeTransform);
		}
	}

	@Override
	public BoundingBox calculateBoundingBox(BoundingBox out) {
		Matrix4 original = nodes.get(0).globalTransform.cpy();

		Matrix4 global = transform.cpy().mul(original);

		Vector3 translation = new Vector3(), scale = new Vector3();
		Quaternion quat = new Quaternion();
		global.getTranslation(translation);
		global.getRotation(quat);
		global.getScale(scale);

		nodes.get(0).translation.set(translation);
		nodes.get(0).rotation.set(quat);
		nodes.get(0).scale.set(scale);

		calculateTransforms();
		super.calculateBoundingBox(out);
		nodes.get(0).translation.set(original.getTranslation(new Vector3()));
		nodes.get(0).rotation.set(original.getRotation(new Quaternion()));
		nodes.get(0).scale.set(original.getScale(new Vector3()));
		calculateTransforms();
		return out;
	}

	public float calculateEffectiveBoundingBox() {
		float time = System.currentTimeMillis() / 1000.0f;
		Matrix4 original = this.transform.cpy();
		Vector3 tmpMin = new Vector3(), tmpMax = new Vector3();
		boundingBox.getMin(tmpMin);
		boundingBox.getMax(tmpMax);
		Vector3 tmpVec = tmpMax.cpy().sub(tmpMin);
		float minVol = Math.abs(tmpVec.x * tmpVec.y * tmpVec.z);
		optimalVolumeAngle = 0;
		rotatedBoundingBox.set(boundingBox);
		rotatedBoundingBox.setTransform(new Matrix4().setToTranslation(getPosition())
				.rotate(Vector3.Y, -optimalVolumeAngle).translate(getPosition().cpy().scl(-1)));
		for (int i = 1; i < 180; i++) {
			rotate(Vector3.Y, 0.5f);
			calculateBoundingBox(boundingBox);
			boundingBox.getMin(tmpMin);
			boundingBox.getMax(tmpMax);
			tmpVec = tmpMax.cpy().sub(tmpMin);
			float newVolu = Math.abs(tmpVec.x * tmpVec.y * tmpVec.z);
			if (newVolu < minVol) {
				optimalVolumeAngle = i * 0.5f;
				minVol = newVolu;
				rotatedBoundingBox.set(boundingBox);
				rotatedBoundingBox.setTransform(new Matrix4().setToTranslation(getPosition())
						.rotate(Vector3.Y, -optimalVolumeAngle).translate(getPosition().cpy().scl(-1)));
			}
		}
		this.transform.set(original);
		calculateBoundingMesh();
		return optimalVolumeAngle;

	}

	void calculateBoundingMesh() {
		Vector3 min = new Vector3();
		rotatedBoundingBox.getMin(min);
		Vector3 max = new Vector3();
		rotatedBoundingBox.getMax(max);
		float vertices[] = {
				// bottom face
				min.x, min.y, min.z, min.x, min.y, max.z, min.x, min.y, max.z, max.x, min.y, max.z, max.x, min.y, max.z,
				max.x, min.y, min.z, max.x, min.y, min.z, min.x, min.y, min.z,
				// top face
				min.x, max.y, min.z, min.x, max.y, max.z, min.x, max.y, max.z, max.x, max.y, max.z, max.x, max.y, max.z,
				max.x, max.y, min.z, max.x, max.y, min.z, min.x, max.y, min.z,
				// side faces
				min.x, min.y, min.z, min.x, max.y, min.z, min.x, min.y, max.z, min.x, max.y, max.z, max.x, min.y, max.z,
				max.x, max.y, max.z, max.x, min.y, min.z, max.x, max.y, min.z };
		if(boundingMesh != null)
			boundingMesh.dispose();
		
		boundingMesh = new Mesh(true, vertices.length, 0, // static mesh with 4 vertices and no indices
				new VertexAttribute(Usage.Position, 3, ShaderProgram.POSITION_ATTRIBUTE));
		boundingMesh.setVertices(vertices);
	}

	public float getOptimalVolumeAngle() {
		return optimalVolumeAngle;
	}

	public void setOptimalVolumeAngle(float optimalVolumeAngle) {
		this.optimalVolumeAngle = optimalVolumeAngle;
	}


	public RotatedBoundingBox getRotatedBoundingBox() {
		return rotatedBoundingBox;
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();
	}
	public void cleanUp() {
		if(sketchModel!=null)sketchModel.dispose();
		if(sketchInstance!=null)sketchInstance.cleanUp();
		worldObject=null;
		position=null;
		dimensions=null;
		sketchModel=null;
		sketchInstance=null;
		if(boundingMesh != null)
			boundingMesh.dispose();
	}

	@Override
	public String toString() {
		return "Object3D [worldObject=" + worldObject.getName() + ", radius=" + radius + ", position=" + position
				+ ", dimensions=" + dimensions + "]";
	}

	public void drawQuotations(Batch batch, Camera camera) {
		List<Vector3> corners = getRotatedBoundingBox().getCorners();
		Vector3 furthest = null;
		for(Vector3 corner: corners) {
			if(furthest == null ||
					furthest.dst2(camera.position) < corner.dst2(camera.position)) {
				furthest = corner;
			}
		}
		
		List<LineSegment> edges = getRotatedBoundingBox().getEdges();
		//remove edges connected to the hidden corner
		boolean stop;
		do {
			stop = true;
			for(LineSegment lineSegment: edges) {
				if(lineSegment.startOrEndWith(furthest, 0.001f)) {
					edges.remove(lineSegment);
					stop = false;
					break;
				}
			}
		}while(!stop);
		//remove quotations from hidden edges
		
				
		if(shapeRenderer == null)
			shapeRenderer = new ShapeRenderer();
		Vector3 furthestFromCamera = null;
		for(LineSegment segment: edges) {
			shapeRenderer.begin(ShapeType.Filled);
			shapeRenderer.setColor(Color.BLUE);
			Vector3 v0 = camera.project(segment.getV0().cpy());
			Vector3 v1 = camera.project(segment.getV1().cpy());
			Vector3 f = segment.getV0().dst2(camera.position) > segment.getV1().dst2(camera.position)?
					segment.getV0().cpy():segment.getV1().cpy();
			if(furthestFromCamera == null || furthestFromCamera.dst2(camera.position) < f.dst2(camera.position))
				furthestFromCamera = f;
			shapeRenderer.circle(v0.x, v0.y, 4);
			shapeRenderer.circle(v1.x, v1.y, 4);
			shapeRenderer.line(v0, v1);
			shapeRenderer.end();
			String length = String.format("%.3f", v0.dst(v1));
			Vector3 v = v1.cpy().add(v0).scl(0.5f);
			float angle = new Vector2(v1.x - v0.x, v1.y - v0.y).angle() % 180;
			Matrix4 transform = new Matrix4();
			//transform.setToRotation(Vector3.Z, 45);
			transform.set(v.cpy().scl(1,1,0), new Quaternion(Vector3.Z, angle));
			String measure = String.format("%.0f",segment.len() * 1000);
			batch.setTransformMatrix(transform);
			batch.begin();
			//AssetFont.getInstance().getSmallFont().draw(batch, measure, 0, 0);
			AssetFont.getInstance().getFont().draw(batch, Math.round(angle) + "", 0, 0, 0, 0, false, Color.BLACK);
			batch.end();
			transform.setToTranslation(new Vector3());
			transform.toString();
		}
		camera.project(furthestFromCamera);
//		shapeRenderer.setColor(Color.RED);
//		shapeRenderer.circle(furthestFromCamera.x, furthestFromCamera.y,10);
//		shapeRenderer.end();
	}

	public void addOns(Object3D... addons) {
		for(var addon: addons) {
			this.addons.add(addon);
		}
	}

	public List<Object3D> getAddons() {
		return addons;
	}
}
