package gdxapp.object3d;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.assets.AssetManager;
import com.badlogic.gdx.assets.loaders.resolvers.AbsoluteFileHandleResolver;
import com.badlogic.gdx.files.FileHandle;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.utils.Disposable;
import com.badlogic.gdx.utils.IntIntMap;
import dressing.config.WorkspaceConfiguration;
import dressing.events.EventDriver;
import dressing.io.IOUtilities;
import dressing.model.ModelProvider;
import dressing.model.ProjectManager;
import dressing.model.persistence.mappers.MapperProvider;
import gdxapp.scenes.SceneEvent;
import jakarta.xml.bind.JAXBException;

public class ModelsLoader implements Disposable{
	
	private final String TAG = ModelsLoader.class.getName();
	private AssetManager assetManager;
	private List<ModeledObject> objects = new ArrayList<ModeledObject>();
	private  static ModelsLoader instance;
	
	private ModelsLoader() {
		assetManager = new AssetManager(new AbsoluteFileHandleResolver());
	}
	
	
	public void loadModels() {
		File dir = new File(WorkspaceConfiguration.getModelsDir());
		File[] dirs = dir.listFiles(new FilenameFilter() {
			@Override
			public boolean accept(File arg0, String arg1) {
				return arg0.isDirectory();
			}
		});
		for(File fileX: dirs) {
			try {
				ObjectModel model = new ObjectModel(fileX.getAbsolutePath());
				ModelProvider.addModelObject(model);
				ModeledObject wObject = WorldObjectFactory.getFactory().createObjectFromModel(model);
				File[] files = new File(fileX.getAbsolutePath()).listFiles();
				for(File fiX: files) {
					if(fiX.getName().equals("preview.png")) {
						wObject.getProperties().put("preview", fiX.getAbsolutePath());
						break;
					}
				}
				objects.add(wObject);
			}catch(Exception e){
				e.printStackTrace();
				continue;
			}
		}
		ModelProvider.setDesignObjects(objects);
		EventDriver.getDriver().deliverEvent(SceneEvent.UPDATE_DATAMODEL, objects);
	}
	
	//load the model residing in the folder specified by the path
	public ObjectModel loadObjectModel(String path) {
		ModelInfo adj = null;
		Model model = null;
		File modelFile = null;
		File adjustmentFile = null;
		File[] files = new File(path).listFiles();
		for(File fileX: files) {
			if(fileX.getName().equals("model.g3dj"))
				modelFile = fileX;
			if(fileX.getName().equals("info.xml"))
				adjustmentFile = fileX;
			if(modelFile != null && adjustmentFile!= null)
				break;
		}		
		if(adjustmentFile != null) {
				adj = MapperProvider.getXmlMapper().fromXml(adjustmentFile, ModelInfo.class);			
		}
		return new ObjectModel(model, adj);
	}
	
	public Model loadModel(File file) {
		Model model = null;
		String pathModel = IOUtilities.convertPathToFileHandlerPath(file.getAbsolutePath());
		assetManager.load(pathModel, Model.class);
		assetManager.finishLoading();
		model =  assetManager.get(pathModel, Model.class);
		return model;
	}
	
	public WorldObject findObject(String name) {
		WorldObject wObject = null;
		for (WorldObject worldObject : objects) {
			if(worldObject.getName().equals(name)){
				wObject = worldObject;
				break;
			}
		}
		return wObject;
	}
	
	
	public WorldObject findObject(UUID id) {
		WorldObject wObject = null;
		UUID uuid = null;
		for(WorldObject worldObject : objects) {
			try {
				uuid = worldObject.getModel().getInfo().getUuid();
				if(uuid.equals(id)) {
					wObject = worldObject;
					break;
				}
			}catch (NullPointerException e) {
				continue;
			}
			
		}
		return wObject;
	}
	
	
	
	
	public List<ModeledObject> getObjects() {
		return objects;
	}

	public void setObjects(List<ModeledObject> objects) {
		this.objects = objects;
	}
	
	public AssetManager getAssetManager() {
		return assetManager;
	}
	
	

	public static ModelsLoader getInstance() {
		if(instance == null)
			instance = new ModelsLoader();
		return instance;
	}

	@Override
	public void dispose() {
		assetManager.dispose();
	}

}
