package api.provider;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import api.graphics.Model;
import api.graphics.ModelInstance;
import api.utils.RessourceLoader;
import org.frs.serialization.utilities.JSONUtilities;
import org.joml.Matrix4f;
import org.joml.Vector3f;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;


public class ModelProvider {

	public static String root = System.getenv("ProgramData") + File.separator + "supercad" + File.separator + "ressources";
	private static final String dataFilePath = "models.json";
	
	private ArrayList<ModelInfo> modelInfos = new ArrayList<ModelInfo>();
	
	private static ModelProvider provider;
	
	private ModelProvider() {
		loadModels();
	}
	
	public void loadModels() {
		
		String filePath = root + File.separator + dataFilePath;
		File file = new File(filePath);
		String jsonStr = null;
		try {		
			FileInputStream fis = new FileInputStream(file);
			var data = fis.readAllBytes();
			fis.close();
			jsonStr = new String(data);
			if(jsonStr.length() > 0) {
				JSONParser parser = new JSONParser();
				JSONArray modelsJSON =  (JSONArray) parser.parse(jsonStr);
				for(var jsonObj: modelsJSON) {
					JSONObject obj = (JSONObject) jsonObj;
					String name = (String) obj.getOrDefault("name", "");
					String label = (String) obj.getOrDefault("label", "");
					String path = (String) obj.getOrDefault("path", "");
					String image = (String) obj.getOrDefault("image", "");
					float[] scale =  JSONUtilities.toFloatArray( (JSONArray) obj.getOrDefault("scale", new float[] {1,1,1}));
					ModelInfo modelInfo = new ModelInfo();
					modelInfo.setName(name);
					modelInfo.setLabel(label);
					modelInfo.setModelPath(path);
					modelInfo.setImagePath(image);
					modelInfo.setScale(new Vector3f(scale));
					modelInfos.add(modelInfo);					
				}

			}
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			return;
		}
	}
	
	public ModelInfo findModelByName(String modelName) {
		ModelInfo info = null;
		for(var infoX: this.modelInfos) {
			if(infoX.getName().equals(modelName)) {
				info = infoX;
				break;
			}
		}
		return info;
	}
	
	public ModelInstance createInstanceOf(String modelName) {
		ModelInstance instance = null;
		ModelInfo info = findModelByName(modelName);
		if(info != null) {
			instance = new ModelInstance(info.getModel());
			var bbox = instance.calculateBoundingBox();
			Vector3f center =  bbox.getCenter();
			Vector3f size = bbox.getSize();
			Matrix4f transform = new Matrix4f().scale(new Vector3f(info.getScale()).div(size)).translate(center.mul(-1));
			instance.setTransform(transform);
			instance.setQuotable(true);
			instance.setMoveable(true);
			instance.setDeletable(true);
		}
		return instance;	
	}
	
	



	public ArrayList<ModelInfo> getModelInfos() {
		return modelInfos;
	}

	public static ModelProvider getProvider() {
		if(provider == null)
			provider = new ModelProvider();
		return provider;
	}

	public static void setProvider(ModelProvider provider) {
		ModelProvider.provider = provider;
	}

	
	
	
	
	
}
