package supercad.runtime;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Stream;

import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.wiring.BundleWiring;
import org.postgresql.shaded.com.ongres.scram.common.bouncycastle.pbkdf2.Pack;

public class ReflectionUtils {

	public static final int PRIVATE_MODIFIER = 2;
	public static final int PROTECTED_MODIFIER = 4;
	public static final int DEFAULT_MODIFIER = 0;
	public static final int PUBLIC_MODIFIER = 1;
	private static final String[] srcRootPackages = { "dressing", "gdxapp", "geometry", "reporting" };
	private static HashSet<Package> pckgs;
	private static Class[] PRIMITIVES = new Class[] { Boolean.class, Byte.class, Character.class, Short.class,
			Integer.class, Long.class, Float.class, Double.class };
	private static boolean srcLoaded;

	public static ArrayList<Class> srcClasses = new ArrayList<Class>();

	// load the classes defined in the Dressing plugin src path excluding classes
	// coming from jars and other build paths
	public static void loadSrcPathClasses() {
		pckgs = new HashSet<Package>();
		if (!srcLoaded) {
			Bundle bundle = Platform.getBundle("Dressing");
			if (bundle == null)
				return;
			BundleWiring bundleWiring = bundle.adapt(BundleWiring.class);
			for (String pckge : srcRootPackages) {
				Collection<String> classNames = bundleWiring.listResources(pckge, "*.class",
						BundleWiring.LISTRESOURCES_RECURSE);
				for (String className : classNames) {
					if (className.contains("$"))
						continue;
					String fullyQualifiedName = className.replace("/", ".").substring(0, className.length() - 6);
					try {
						Class<?> clazz = bundle.loadClass(fullyQualifiedName);
						pckgs.add(clazz.getPackage());
						srcClasses.add(clazz);
					} catch (ClassNotFoundException e) {
						e.printStackTrace();
					}
				}
			}
			srcLoaded = true;
		}
	}

	public static ArrayList<Class> getDeclaredClasses() {
		ArrayList<Class> classes = new ArrayList<Class>();
		for (Class clazz : ReflectionUtils.getSrcClasses()) {
			classes.add(clazz);
		}
		return classes;
	}

	public static ArrayList<Class> getDeclaredClasses(Package pckg) {
		ArrayList<Class> pckgClasses = new ArrayList<Class>();
		for (Class clazz : srcClasses) {
			if (clazz.getPackage().equals(pckg))
				pckgClasses.add(clazz);
		}
		return pckgClasses;
	}

	public static void getAllFields(Class clazz, ArrayList<Field> out, int... modifiers) {
		ArrayList<Integer> mods = new ArrayList<Integer>();
		if (modifiers.length < 1)
			modifiers = new int[] { 0, 1, 2, 3, 4 };
		for (int x : modifiers) {
			mods.add(Integer.valueOf(x));
		}
		Field[] fields = clazz.getDeclaredFields();
		for (Field fieldX : fields) {
			Integer fieldModifier = fieldX.getModifiers();
			if (mods.contains(fieldModifier))
				out.add(fieldX);
		}
		Class superClass = clazz.getSuperclass();
		if (superClass != null && srcClasses.contains(superClass))
			getAllFields(clazz.getSuperclass(), out, ReflectionUtils.PUBLIC_MODIFIER,
					ReflectionUtils.PROTECTED_MODIFIER);
	}

	public static void getFields(Class clazz, ArrayList<Field> out, int... modifiers) {
		ArrayList<Integer> mods = new ArrayList<Integer>();
		if (modifiers.length < 1)
			modifiers = new int[] { 0, 1, 2, 3, 4 };
		for (int x : modifiers) {
			mods.add(Integer.valueOf(x));
		}
		Field[] fields = clazz.getDeclaredFields();
		for (Field fieldX : fields) {
			Integer fieldModifier = fieldX.getModifiers();
			if (mods.contains(fieldModifier))
				out.add(fieldX);
		}
		Class superClass = clazz.getSuperclass();
		if (superClass != null)
			getFields(clazz.getSuperclass(), out, ReflectionUtils.PUBLIC_MODIFIER, ReflectionUtils.PROTECTED_MODIFIER);
	}

	public static ArrayList<Field> getNonTransientField(Object obj) {
		ArrayList<Field> fields = new ArrayList<Field>();
		ReflectionUtils.getFields(obj.getClass(), fields, 0, 1, 2, 3, 4);
		ArrayList<Field> staticAndTransient = new ArrayList<Field>();
		for (Field field : fields) {
			int mod = field.getModifiers();
			if (Modifier.isStatic(mod) || Modifier.isTransient(mod))
				staticAndTransient.add(field);
		}
		fields.removeAll(staticAndTransient);
		return fields;
	}

	public static boolean isPrimitive(Class clazz) {
		if(clazz.isPrimitive())
			return true;
		for (Class primitive : PRIMITIVES) {
			if (clazz == primitive)
				return true;
		}
		return false;
	}

	public static boolean isSimpleType(Class clazz) {
		if (clazz.isArray())
			return false;
		if (Collection.class.isAssignableFrom(clazz))
			return false;
		if (Map.class.isAssignableFrom(clazz))
			return false;
		return true;
	}

	public static Class getClass(String name) {
		switch (name) {
		case "float":
			return Float.class;
		case "int":
			return Integer.class;
		case "long":
			return Long.class;
		case "double":
			return Double.class;
		case "short":
			return Short.class;
		case "boolean":
			return Boolean.class;
		case "byte":
			return Byte.class;
		default:
			try {
				return Class.forName(name);
			} catch (ClassNotFoundException e) {
				e.printStackTrace();
			}
		}
		return null;
	}

	public static void getMethods(Class clazz, ArrayList<Method> out, int... modifiers) {
		ArrayList<Integer> mods = new ArrayList<Integer>();
		if (modifiers.length < 1)
			modifiers = new int[] { 0, 1, 2, 3, 4 };
		for (int x : modifiers) {
			mods.add(Integer.valueOf(x));
		}
		Method[] methods = clazz.getDeclaredMethods();
		for (Method methodX : methods) {
			Integer fieldModifier = methodX.getModifiers();
			if (mods.contains(fieldModifier))
				out.add(methodX);
		}
		Class superClass = clazz.getSuperclass();
		if (superClass != null)
			getMethods(superClass, out, ReflectionUtils.PUBLIC_MODIFIER, ReflectionUtils.PROTECTED_MODIFIER);
	}

	public static Method findSetter(Field field) {
		ArrayList<Method> methods = new ArrayList<Method>();
		getMethods(field.getClass(), methods, ReflectionUtils.PUBLIC_MODIFIER);
		for (Method methodX : methods) {
			if (methodX.getParameterCount() == 1) {
				Parameter param = methodX.getParameters()[0];
				boolean paramMatch = param.getType() == field.getType();
				if (paramMatch && methodX.getName().toLowerCase().equals("set" + field.getName().toLowerCase()))
					return methodX;
			}
		}
		return null;
	}

	public static Constructor getNoArgsConstructor(Class clazz) {
		Constructor constructor = null;
		try {
			constructor = clazz.getConstructor(new Class[0]);
		} catch (SecurityException | NoSuchMethodException e) {
			e.printStackTrace();
		}
		return constructor;
	}

	public static boolean isCollection(Object object) {
		return object.getClass().isArray() || (object instanceof Collection) || (object instanceof Map);
	}

	public static String traceHierarchy(Class type) {
		ArrayList<Class> hierarchy = new ArrayList<Class>();
		Class tmp = type;
		do {
			hierarchy.add(tmp);
			tmp = tmp.getSuperclass();
		} while (tmp != null);
		String str = type.getName();
		for (int i = 1; i < hierarchy.size(); i++) {
			str += "-->" + hierarchy.get(i);
		}
		return str;
	}

	public static boolean isSrcLoaded() {
		return srcLoaded;
	}

	public static void setSrcLoaded(boolean srcLoaded) {
		ReflectionUtils.srcLoaded = srcLoaded;
	}

	public static ArrayList<Class> getSrcClasses() {
		return srcClasses;
	}

	public static void setSrcClasses(ArrayList<Class> srcClasses) {
		ReflectionUtils.srcClasses = srcClasses;
	}

	public static String[] getRootPackgs() {
		return srcRootPackages;
	}

	public static HashSet<Package> getPckges() {
		return pckgs;
	}

	public static boolean isNullOrEmptyCollection(Object value) {
		if (value == null)
			return true;
		if (isCollection(value)) {
			if (value.getClass().isArray()) {
				return Array.getLength(value) < 1;
			}
			if (value instanceof Map) {
				Map map = (Map) value;
				return map.values().isEmpty();
			}
			if (value instanceof Collection) {
				Collection col = (Collection) value;
				return col.size() < 1;
			}
		}
		return false;
	}
	
	public static <T> List<T> getList(Object obj, Class<T> componentType) {
		if(obj.getClass().isArray()) {
			boolean stop = false;
			int index = 0;
			ArrayList<T> list = new ArrayList<T>();
			while(!stop) {
				try {
					T element = (T) Array.get(obj, index++);
					list.add(element);
				}catch (IndexOutOfBoundsException e) {
					stop = true;
				}
			}
			return list;
		}
		return null;
	}
}
