package api.ui;

import java.io.File;
import java.util.HashMap;
import java.util.function.Consumer;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.events.ShellListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGBA;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.opengl.GLCanvas;
import org.eclipse.swt.opengl.GLData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.wb.swt.SWTResourceManager;
import org.joml.Vector2i;
import org.joml.Vector3f;
import org.lwjgl.opengl.GL;

import api.backend.ApplicationContext;
import api.events.CustomEvent;
import api.events.EventBus;
import api.events.EventHandler;
import api.events.EventType;
import api.graphics.Camera.CAMERA_MODE;
import api.graphics.Camera.PROJECTION_MODE;
import api.graphics.ModelInstance;
import api.graphics.ObjectController.OPERATION;
import api.provider.ModelProvider;
import api.utils.RessourceLoader;
import gdxapp.Commun.MeasuresTaker;

public class EditorContainer extends Composite implements EventHandler {

	private GLCanvas canvas;
	private ToolBar mainMenuToolBar; 
	private RightMenuToolBarManager rightMenuToolBarManager;

	private StackLayout rightMenuStackLayout;
	private PlumbingMenu plumbingMenu;
	private ModelLibrary library;
	private SelectionMenu selectionMenu;

	SurfaceAnchors surfaceAnchors;
	private Composite rightBarContainer;



	public EditorContainer(Composite parent, int style) {
		super(parent, style);
		GridLayout gridLayout = new GridLayout(1, true);
		gridLayout.verticalSpacing = 0;
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		setLayout(gridLayout);

		Composite topComposite = new Composite(this, SWT.NONE);
		GridLayout gl_topComposite = new GridLayout(1, false);
		gl_topComposite.marginWidth = 0;
		gl_topComposite.verticalSpacing = 0;
		gl_topComposite.marginHeight = 0;
		gl_topComposite.horizontalSpacing = 0;
		topComposite.setLayout(gl_topComposite);
		topComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		

		Composite mainComposite = new Composite(this, SWT.NONE);
		GridLayout gl_mainComposite = new GridLayout(2, false);
		gl_mainComposite.horizontalSpacing = 0;
		gl_mainComposite.marginHeight = 0;
		gl_mainComposite.marginWidth = 0;
		gl_mainComposite.verticalSpacing = 0;
		mainComposite.setLayout(gl_mainComposite);
		mainComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		
		
		
		Composite canvasContainer = new Composite(mainComposite, SWT.NONE);
		canvasContainer.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		canvasContainer.setLayout(null);
		createCanvasPart(canvasContainer);
		surfaceAnchors = new SurfaceAnchors(canvasContainer);

		
		Composite sideContainer = new Composite(mainComposite, SWT.NONE);
		sideContainer.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, true, 1, 1));
		GridLayout gl_sideContainer = new GridLayout(1, true);
		gl_sideContainer.marginHeight = 0;
		gl_sideContainer.verticalSpacing = 0;
		gl_sideContainer.marginWidth = 0;
		gl_sideContainer.horizontalSpacing = 0;
		sideContainer.setLayout(gl_sideContainer);
		RightMenuToolBarManager toolBarManager2 = new RightMenuToolBarManager(sideContainer);
		rightBarContainer = new Composite(sideContainer, SWT.NONE);
		rightBarContainer.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, true, 1, 1));
		rightBarContainer.setBackground(SWTResourceManager.getColor(SWT.COLOR_WIDGET_DARK_SHADOW));
		rightMenuStackLayout = new StackLayout();
		rightBarContainer.setLayout(rightMenuStackLayout);
		plumbingMenu = new PlumbingMenu(rightBarContainer, SWT.NONE);
		library = new ModelLibrary(rightBarContainer, SWT.NONE);
		selectionMenu = new SelectionMenu(rightBarContainer, SWT.NONE);
		
		createMainMenuToolBar(topComposite);
		
	
		Composite bottomComposite = new Composite(this, SWT.NONE);
		bottomComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		
		setLayoutForSelection();

		subscribeToEvent();
		
		

	}

	private void createCanvasPart(Composite canvasContainer) {				
		GLData data = new GLData();
		data.doubleBuffer = true;
		canvas = new GLCanvas(canvasContainer, SWT.BORDER, data);
		canvas.setSize(300,300);
		canvas.setCurrent();
		GL.createCapabilities();
		ApplicationContext.setCanvas(canvas);

		DropTarget dropTarget = new DropTarget(canvas, DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_DEFAULT);
		dropTarget.setTransfer(TextTransfer.getInstance());
		dropTarget.addDropListener(new DropTargetAdapter() {

			@Override
			public void drop(DropTargetEvent event) {
				HashMap<String, Object> eventDetails = new HashMap<String, Object>();
				eventDetails.put("model name", event.data);
				Point canvasOrigin = canvas.toDisplay(0, 0);
				eventDetails.put("location", new Vector2i(event.x - canvasOrigin.x, event.y - canvasOrigin.y));
				EventBus.getInstance().notify(new CustomEvent(EventType.DROP_OBJECT, eventDetails));
			}

		});
		Consumer<ControlEvent>  handler = (ControlEvent e) -> {
			var size = canvasContainer.getSize();
			if(size.x > 0 && size.y > 0) {
				canvas.setSize(canvasContainer.getSize());
				canvas.setLocation(0,0);
			}
		};
		
		ControlListener controlListener = new ControlListener() {
			@Override
			public void controlResized(ControlEvent e) {
				getShell().getDisplay().asyncExec( () -> handler.accept(e)) ;
			}
			
			@Override
			public void controlMoved(ControlEvent e) {
				getShell().getDisplay().asyncExec( () -> handler.accept(e)) ;
			}
		};

		getShell().addControlListener(controlListener);
	}



	private void createMainMenuToolBar(Composite parent) {
		// transformers
		this.mainMenuToolBar = new ToolBar(parent, SWT.NONE);
		mainMenuToolBar = new ToolBar(parent, SWT.FLAT | SWT.RIGHT);
		mainMenuToolBar.setBackground(SWTResourceManager.getColor(SWT.COLOR_GRAY));
		mainMenuToolBar.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, true, false));
		//transform operations
		addTransformOperationItems();
		ToolItem seperator = new ToolItem(mainMenuToolBar, SWT.SEPARATOR);		
		//tools
		addToolsItem();
		//camera
		ToolItem seperator2 = new ToolItem(mainMenuToolBar, SWT.SEPARATOR);
		addCameraItems();
	}

	private void addCameraItems() {
		ToolItem persProjItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		persProjItem.setText("perspective");
		persProjItem.setSelection(true);
		persProjItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if(persProjItem.getSelection()) {
					HashMap<String, Object> details = new HashMap<String, Object>();
					details.put("proj_mode", PROJECTION_MODE.PERSPECTIVE);
					EventBus.getInstance().notify(new CustomEvent(EventType.PROJECTION_MODE_CHANGED, details));
				}
			}
		});
		
		ToolItem orthoProjItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		orthoProjItem.setText("orthographic");
		orthoProjItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if(orthoProjItem.getSelection()) {
					HashMap<String, Object> details = new HashMap<String, Object>();
					details.put("proj_mode", PROJECTION_MODE.ORTHOGRAPHIC);
					EventBus.getInstance().notify(new CustomEvent(EventType.PROJECTION_MODE_CHANGED, details));
				}
			
			}
		});
		ToolItem seperator3 = new ToolItem(mainMenuToolBar, SWT.SEPARATOR);
		ToolItem flyModeCameraItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		Image flyImg = new Image(Display.getDefault(), ModelProvider.root + File.separator + "icons/icons8-bee-32.png");
		flyModeCameraItem.setImage(flyImg);
		flyModeCameraItem.setSelection(true);
		flyModeCameraItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if(flyModeCameraItem.getSelection()) {
					HashMap<String, Object> details = new HashMap<String, Object>();
					details.put("camera_mode", CAMERA_MODE.FLY);
					EventBus.getInstance().notify(new CustomEvent(EventType.PROJECTION_MODE_CHANGED, details));
				}
			}
		});
		
		ToolItem orbitCameraItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		Image orbitImg = new Image(Display.getDefault(), ModelProvider.root + File.separator + "icons/icons8-satellites-32.png");
		orbitCameraItem.setImage(orbitImg);
		orbitCameraItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if(orbitCameraItem.getSelection()) {
					HashMap<String, Object> details = new HashMap<String, Object>();
					details.put("camera_mode", CAMERA_MODE.ORBIT);
					EventBus.getInstance().notify(new CustomEvent(EventType.PROJECTION_MODE_CHANGED, details));
				}
			}
		});		
	}

	private void addToolsItem() {
		ToolItem measuresItem = new ToolItem(mainMenuToolBar, SWT.PUSH);
		Image rulerImg = new Image(Display.getDefault(), ModelProvider.root + File.separator + "icons/icons8-ruler-32.png");
		measuresItem.setImage(rulerImg);
		measuresItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				EventBus.getInstance().notify(new CustomEvent(EventType.INVOKE_RULER, null));
			}
			
		});		
	}

	private void addTransformOperationItems() {
		//selcetion item
				ToolItem positionItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
				positionItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						if(positionItem.getSelection()) {
							HashMap< String, Object> details = new HashMap<String, Object>();
							details.put("op", OPERATION.TRANSLATE);
							EventBus.getInstance().notify(new CustomEvent(EventType.TRANSFORM_OP, details) );
						}
					}
				});
				String imagePath = ModelProvider.root + File.separator +  "icons/icons8-drag-32.png";
				Image positionImg = new Image(Display.getDefault(), imagePath);
				positionItem.setImage(positionImg);
				//plumbing menu item
				ToolItem rotationItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
				rotationItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						if(rotationItem.getSelection()) {
							HashMap< String, Object> details = new HashMap<String, Object>();
							details.put("op", OPERATION.ROTATE);
							EventBus.getInstance().notify(new CustomEvent(EventType.TRANSFORM_OP, details) );	
						}
					}
				});
				imagePath = ModelProvider.root + File.separator +  "icons/icons8-3d-rotate-32.png";
				Image rotationImg = new Image(Display.getDefault(), imagePath);
				rotationItem.setImage(rotationImg);
				//electric menu item
				ToolItem scaleItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
				scaleItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						if(scaleItem.getSelection()) {
							HashMap< String, Object> details = new HashMap<String, Object>();
							details.put("op", OPERATION.SCALE);
							EventBus.getInstance().notify(new CustomEvent(EventType.TRANSFORM_OP, details) );
						}
					}
				});
				imagePath = ModelProvider.root + File.separator +  "icons/icons8-stretch-tool-32.png";
				Image scaleImg = new Image(Display.getDefault(), imagePath);
				scaleItem.setImage(scaleImg);
				positionItem.setSelection(true);	
	}

	
	private void subscribeToEvent() {
		subscribe(EventType.OBJECT_SELECTION_TRIGFERED);
		subscribe(EventType.PLUMBING_EDITING_TRIGGERED);
		subscribe(EventType.SELECTION_MENU_TRIGGERED);
	}

	public GLCanvas getCanvas() {
		return canvas;
	}

	public void setCanvas(GLCanvas canvas) {
		this.canvas = canvas;
	}

	@Override
	public void handle(CustomEvent event) {
		EventType type = event.getType();
		if (type == EventType.OBJECT_SELECTION_TRIGFERED) {
			setLayoutForObjectSelection();
		} else if (type == EventType.PLUMBING_EDITING_TRIGGERED) {
			setLayoutForPLumbing();
		}else if(type == EventType.SELECTION_MENU_TRIGGERED) {
			setLayoutForSelection();
		}
	}

	private void setLayoutForPLumbing() {
		getDisplay().asyncExec(() -> {
			rightMenuStackLayout.topControl = plumbingMenu;
			layout(true, true);
		});

	}

	private void setLayoutForObjectSelection() {
		getDisplay().asyncExec(() -> {
			rightMenuStackLayout.topControl = library;
			layout(true, true);
		});
	}
	
	private void setLayoutForSelection() {
		getDisplay().asyncExec(()-> {
			rightMenuStackLayout.topControl = selectionMenu;
			layout(true, true);
		});
	}
}
